<?php

namespace App\Extensions\MarketingBot\System\Services;

use App\Extensions\MarketingBot\System\Models\ShopifyStore;
use App\Extensions\MarketingBot\System\Models\ShopifyProduct;
use App\Extensions\MarketingBot\System\Models\ShopifyOrder;
use App\Extensions\MarketingBot\System\Models\CampaignConversion;
use App\Extensions\MarketingBot\System\Models\MarketingCampaign;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

class ShopifyAnalyticsService
{
    protected ShopifyStore $store;

    public function __construct(ShopifyStore $store)
    {
        $this->store = $store;
    }

    /**
     * Get daily analytics metrics
     */
    public function getDailyMetrics(): array
    {
        $today = Carbon::today();
        $yesterday = Carbon::yesterday();

        return [
            'today' => $this->getMetricsForPeriod($today, Carbon::now()),
            'yesterday' => $this->getMetricsForPeriod($yesterday, $today),
            'comparison' => $this->calculateComparison($today, $yesterday),
        ];
    }

    /**
     * Get metrics for a specific period
     */
    public function getMetricsForPeriod(Carbon $startDate, Carbon $endDate): array
    {
        $orders = $this->store->orders()
            ->whereBetween('shopify_created_at', [$startDate, $endDate])
            ->where('financial_status', '!=', 'cancelled')
            ->get();

        $revenue = $orders->sum('total_price');
        $orderCount = $orders->count();
        $avgOrderValue = $orderCount > 0 ? $revenue / $orderCount : 0;

        return [
            'revenue' => round($revenue, 2),
            'orders' => $orderCount,
            'avg_order_value' => round($avgOrderValue, 2),
            'items_sold' => $orders->sum('total_items'),
        ];
    }

    /**
     * Calculate comparison between periods
     */
    protected function calculateComparison(Carbon $currentStart, Carbon $previousStart): array
    {
        $current = $this->getMetricsForPeriod($currentStart, Carbon::now());
        $previous = $this->getMetricsForPeriod($previousStart, $currentStart);

        return [
            'revenue_change' => $this->calculatePercentageChange($previous['revenue'], $current['revenue']),
            'orders_change' => $this->calculatePercentageChange($previous['orders'], $current['orders']),
            'aov_change' => $this->calculatePercentageChange($previous['avg_order_value'], $current['avg_order_value']),
        ];
    }

    /**
     * Calculate percentage change
     */
    protected function calculatePercentageChange(float $old, float $new): float
    {
        if ($old == 0) {
            return $new > 0 ? 100 : 0;
        }

        return round((($new - $old) / $old) * 100, 1);
    }

    /**
     * Get revenue trend for the last N days
     */
    public function getRevenueTrend(int $days = 30): array
    {
        $data = $this->store->orders()
            ->selectRaw("DATE(shopify_created_at) as date, SUM(total_price) as revenue, COUNT(*) as orders")
            ->where('shopify_created_at', '>=', Carbon::now()->subDays($days))
            ->where('financial_status', '!=', 'cancelled')
            ->groupBy('date')
            ->orderBy('date')
            ->get();

        // Fill in missing dates with zero values
        $dates = [];
        for ($i = $days - 1; $i >= 0; $i--) {
            $date = Carbon::now()->subDays($i)->format('Y-m-d');
            $dates[$date] = [
                'date' => $date,
                'revenue' => 0,
                'orders' => 0,
            ];
        }

        foreach ($data as $row) {
            $dates[$row->date] = [
                'date' => $row->date,
                'revenue' => round($row->revenue, 2),
                'orders' => $row->orders,
            ];
        }

        return array_values($dates);
    }

    /**
     * Get top selling products
     */
    public function getTopProducts(int $limit = 10, ?Carbon $since = null): array
    {
        $since = $since ?? Carbon::now()->subDays(30);

        $orders = $this->store->orders()
            ->where('shopify_created_at', '>=', $since)
            ->where('financial_status', '!=', 'cancelled')
            ->get();

        $productStats = [];

        foreach ($orders as $order) {
            foreach ($order->line_items ?? [] as $item) {
                $productId = $item['product_id'] ?? null;
                if (!$productId) continue;

                if (!isset($productStats[$productId])) {
                    $productStats[$productId] = [
                        'product_id' => $productId,
                        'title' => $item['title'] ?? 'Unknown',
                        'quantity' => 0,
                        'revenue' => 0,
                    ];
                }

                $productStats[$productId]['quantity'] += $item['quantity'] ?? 0;
                $productStats[$productId]['revenue'] += ($item['price'] ?? 0) * ($item['quantity'] ?? 0);
            }
        }

        // Sort by revenue and limit
        usort($productStats, fn($a, $b) => $b['revenue'] <=> $a['revenue']);
        $topProducts = array_slice($productStats, 0, $limit);

        // Enrich with product data
        foreach ($topProducts as &$stat) {
            $product = $this->store->products()
                ->where('shopify_product_id', $stat['product_id'])
                ->first();

            if ($product) {
                $stat['image_url'] = $product->image_url;
                $stat['handle'] = $product->handle;
            }
        }

        return $topProducts;
    }

    /**
     * Get campaign ROI data
     */
    public function getCampaignROI(?Carbon $since = null): array
    {
        $since = $since ?? Carbon::now()->subDays(30);

        $conversions = CampaignConversion::with('campaign')
            ->where('converted_at', '>=', $since)
            ->get()
            ->groupBy('campaign_id');

        $roiData = [];

        foreach ($conversions as $campaignId => $campaignConversions) {
            $campaign = $campaignConversions->first()->campaign;
            if (!$campaign) continue;

            $revenue = $campaignConversions->sum('revenue');
            $orders = $campaignConversions->count();

            // Estimate campaign cost (you might want to add a cost field to campaigns)
            $estimatedCost = 100; // Placeholder

            $roiData[] = [
                'campaign_id' => $campaignId,
                'campaign_name' => $campaign->name,
                'revenue' => round($revenue, 2),
                'orders' => $orders,
                'cost' => $estimatedCost,
                'roi' => $estimatedCost > 0 ? round((($revenue - $estimatedCost) / $estimatedCost) * 100, 1) : 0,
                'roas' => $estimatedCost > 0 ? round($revenue / $estimatedCost, 2) : 0,
            ];
        }

        // Sort by revenue
        usort($roiData, fn($a, $b) => $b['revenue'] <=> $a['revenue']);

        return $roiData;
    }

    /**
     * Get conversion rate
     */
    public function getConversionRate(?Carbon $since = null): float
    {
        $since = $since ?? Carbon::now()->subDays(30);

        // This would need to be enhanced with actual visitor tracking
        // For now, we'll use a simplified version based on orders vs contacts
        $orders = $this->store->orders()
            ->where('shopify_created_at', '>=', $since)
            ->where('financial_status', '!=', 'cancelled')
            ->count();

        // Get unique customers from orders
        $uniqueCustomers = $this->store->orders()
            ->where('shopify_created_at', '>=', $since)
            ->distinct('customer_email')
            ->count('customer_email');

        // Estimate visitors (this is a placeholder - you'd want actual analytics)
        $estimatedVisitors = $uniqueCustomers * 10; // Assume 10% conversion

        return $estimatedVisitors > 0 ? round(($orders / $estimatedVisitors) * 100, 2) : 0;
    }

    /**
     * Get low stock products
     */
    public function getLowStockProducts(int $threshold = 10): array
    {
        return $this->store->products()
            ->where('is_active', true)
            ->where('inventory_quantity', '>', 0)
            ->where('inventory_quantity', '<=', $threshold)
            ->orderBy('inventory_quantity')
            ->limit(20)
            ->get()
            ->map(fn($product) => [
                'id' => $product->id,
                'title' => $product->title,
                'inventory' => $product->inventory_quantity,
                'price' => $product->price,
                'image_url' => $product->image_url,
            ])
            ->toArray();
    }

    /**
     * Get customer lifetime value stats
     */
    public function getCustomerLTVStats(): array
    {
        $customers = $this->store->orders()
            ->selectRaw('customer_email, COUNT(*) as order_count, SUM(total_price) as total_spent')
            ->whereNotNull('customer_email')
            ->groupBy('customer_email')
            ->get();

        $totalCustomers = $customers->count();
        $avgLTV = $totalCustomers > 0 ? $customers->avg('total_spent') : 0;
        $avgOrdersPerCustomer = $totalCustomers > 0 ? $customers->avg('order_count') : 0;

        return [
            'total_customers' => $totalCustomers,
            'avg_ltv' => round($avgLTV, 2),
            'avg_orders_per_customer' => round($avgOrdersPerCustomer, 1),
            'repeat_customer_rate' => $totalCustomers > 0 
                ? round(($customers->where('order_count', '>', 1)->count() / $totalCustomers) * 100, 1)
                : 0,
        ];
    }
}
